<?php
/* --------------------------------------------------------------
   ApiV2HubAuthenticator.inc.php 2018-11-02
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2018 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

use \HubPublic\Http\CurlRequest;

/**
 * Class ApiV2HubAuthenticator
 */
class ApiV2HubAuthenticator extends ApiV2HubAuthenticator_parent
{
	/**
	 * Authorize request with Hub Credentials
	 *
	 * This method overloads the basic authentication to use special hub credentials.
	 * As a fallback this overload calls the basic auth of the parent.
	 *
	 * @param string $controllerName Name of the parent controller for this api call.
	 *
	 * @throws HttpApiV2Exception If request does not provide the "Authorization" header or if the
	 *                            credentials are invalid.
	 *
	 * @throws InvalidArgumentException If the username or password values are invalid.
	 */
	public function authorize($controllerName)
	{
		// Check if Hub authentication is allowed
		if(gm_get_conf('GAMBIO_HUB_ALLOW_REST_ACTIONS') === 'false')
		{
			parent::authorize($controllerName);
			
			return;
		}
		
		try
		{
			// Check if this is a Hub IP. 
			$this->validateCallbackRequest();
			
			// Check if the client key is valid. 
			if($this->api->request->headers->get('X-CLIENT-KEY') !== gm_get_conf('GAMBIO_HUB_CLIENT_KEY'))
			{
				throw new RuntimeException('Forbidden', 403);
			}
			
			// Check if the requested REST API action is allowed.
			$this->validateRestActions();
		}
		catch(\RuntimeException $exception)
		{
			parent::authorize($controllerName);
		}
	}
	
	
	/**
	 * Validate request IP
	 */
	protected function validateCallbackRequest()
	{
		$response = $this->getIpList();
		
		if($response->getStatusCode() === 200)
		{
			$ipList = @json_decode($response->getBody(), true);
			
			if(!is_array($ipList))
			{
				$response = $this->getIpList(true); // retry without cache
				$ipList   = @json_decode($response->getBody(), true);
			}
			
			if(is_array($ipList))
			{
				try
				{
					$this->isIpValid($ipList);
				}
				catch(RuntimeException $exception)
				{
					$response = $this->getIpList(true); // retry without cache
					
					if($response->getStatusCode() === 200)
					{
						$ipList = @json_decode($response->getBody(), true);
						$this->isIpValid($ipList);
					}
				}
			}
		}
	}
	
	
	/**
	 * Checks the IP of this request.
	 *
	 * Only the Hub IP is allowed to make these kind of requests.
	 *
	 * @param array $ipList
	 */
	protected function isIpValid(array $ipList)
	{
		$ip = isset($_SERVER['HTTP_X_FORWARDED_FOR']) ? $_SERVER['HTTP_X_FORWARDED_FOR'] : $_SERVER['REMOTE_ADDR'];
		
		$valid = false;
		foreach($ipList as $hubIp)
		{
			if($hubIp === '*' || strpos($ip, $hubIp) === 0)
			{
				$valid = true;
				break;
			}
		}
		
		// Check with IP whitelist (comma separated IP values).
		if (!$valid && defined('MODULE_PAYMENT_GAMBIO_HUB_IP_WHITELIST'))
		{
			$ipWhitelist = explode(',', MODULE_PAYMENT_GAMBIO_HUB_IP_WHITELIST);
			
			foreach($ipWhitelist as $ipWhitelistEntry) {
				if(empty($ipWhitelistEntry))
				{
					continue;
				}
				
				// Will also match partial IP values like "192.168.0".
				if ($ipWhitelistEntry === '*' || strpos($ip, trim($ipWhitelistEntry)) !== false)
				{
					$valid = true;
					break;
				}
			}
		}
		
		if(!$valid)
		{
			throw new RuntimeException('Forbidden', 403);
		}
	}
	
	/**
	 * Returns the trusted Hub IP list.
	 *
	 * @param bool $bypassCacheValue Bypass the caching mechanism.
	 *
	 * @return \HubPublic\ValueObjects\HttpResponse|mixed
	 */
	protected function getIpList($bypassCacheValue = false)
	{
		$response = $this->getCacheValue('hub-ip-list');
		
		if($response !== null)
		{
			$response = unserialize($response);
		}
		
		if($response === null || $bypassCacheValue || !is_object($response))
		{
			$curlRequest = new CurlRequest();
			$response    = $curlRequest->setUrl($this->getIpListUrl())->execute();
			
			if($response->getStatusCode() === 200)
			{
				// serialize value for compatibility with cache of old connector
				$this->setCacheValue('hub-ip-list', serialize($response), new DateTime('tomorrow'));
			}
		}
		
		return $response;
	}
	
	
	/**
	 * Returns the IP list URL.
	 *
	 * @return string
	 */
	protected function getIpListUrl()
	{
		$url = 'https://core-api.gambiohub.com/trust/hub_hosts.json'; // Default value. 
		
		if(defined('MODULE_PAYMENT_GAMBIO_HUB_IP_LIST_URL'))
		{
			$url = MODULE_PAYMENT_GAMBIO_HUB_IP_LIST_URL;
		}
		
		return $url;
	}
	
	
	/**
	 * Validate request REST API action.
	 */
	protected function validateRestActions()
	{
		$response = $this->getRestActions();
		
		if($response->getStatusCode() === 200)
		{
			$restActions = @json_decode($response->getBody(), true);
			
			if(!is_array($restActions))
			{
				$response    = $this->getRestActions(true); // retry without cache
				$restActions = @json_decode($response->getBody(), true);
			}
			
			if(is_array($restActions))
			{
				try
				{
					$this->isActionValid($restActions);
				}
				catch(RuntimeException $exception)
				{
					$response = $this->getRestActions(true); // retry without cache
					
					if($response->getStatusCode() === 200)
					{
						$restActions = @json_decode($response->getBody(), true);
						$this->isActionValid($restActions);
					}
				}
			}
		}
	}
	
	
	/**
	 * Checks the REST action of this request.
	 *
	 * Only the certain actions are allowed.
	 *
	 * @param array $restActions
	 */
	protected function isActionValid(array $restActions)
	{
		$resource = substr($this->api->request->getResourceUri(), 3);
		$method   = strtoupper($this->api->request->getMethod());
		
		if(is_array($restActions) && isset($restActions[$method]))
		{
			$valid = false;
			foreach($restActions[$method] as $allowedAction)
			{
				if($allowedAction === '*' || strpos($resource, $allowedAction) === 0)
				{
					$valid = true;
					break;
				}
			}
			
			if(!$valid)
			{
				throw new RuntimeException('Forbidden', 403);
			}
		}
	}
	
	/**
	 * Returns the allowed REST API actions.
	 *
	 * @param bool $bypassCacheValue Bypass the caching mechanism.
	 *
	 * @return \HubPublic\ValueObjects\HttpResponse|mixed
	 */
	protected function getRestActions($bypassCacheValue = false)
	{
		$response = $this->getCacheValue('hub-rest-actions');
		
		if($response !== null)
		{
			$response = unserialize($response);
		}
		
		if($response === null || $bypassCacheValue || !is_object($response))
		{
			$curlRequest = new CurlRequest();
			$response    = $curlRequest->setUrl($this->getRestActionsUrl())->execute();
			
			if($response->getStatusCode() === 200)
			{
				// serialize value for compatibility with cache of old connector
				$this->setCacheValue('hub-rest-actions', serialize($response), new DateTime('tomorrow'));
			}
		}
		
		return $response;
	}
	
	
	/**
	 * Returns the IP list url.
	 *
	 * @return string
	 */
	protected function getRestActionsUrl()
	{
		$url = 'https://core-api.gambiohub.com/trust/rest_actions.json'; // Default value. 
		
		if(defined('MODULE_PAYMENT_GAMBIO_HUB_REST_ACTIONS_URL'))
		{
			$url = MODULE_PAYMENT_GAMBIO_HUB_REST_ACTIONS_URL;
		}
		
		return $url;
	}
	
	
	/**
	 * Returns cache value identified by given key.
	 *
	 * @return mixed
	 */
	protected function getCacheValue($key)
	{
		$value = null;
		
		$cacheFilePath = $this->getCacheFilePath($key);
		
		if(file_exists($cacheFilePath))
		{
			$value = unserialize(file_get_contents($cacheFilePath));
		}
		
		if(isset($value['__expires']))
		{
			$value = $value['__expires'] > new DateTime ? $value['value'] : null;
		}
		
		return $value;
	}
	
	
	/**
	 * Stores value in cache identified by given key.
	 *
	 * @param                $key
	 * @param                $value
	 * @param \DateTime|null $expires
	 */
	protected function setCacheValue($key, $value, DateTime $expires = null)
	{
		$cacheFilePath = $this->getCacheFilePath($key);
		
		if($expires !== null)
		{
			$value = [
				'__expires' => $expires,
				'value'     => $value
			];
		}
		
		file_put_contents($cacheFilePath, serialize($value));
	}
	
	
	/**
	 * Returns cache file path.
	 *
	 * @param string $key
	 *
	 * @return string
	 */
	protected function getCacheFilePath($key)
	{
		$cacheFilePath = DIR_FS_CATALOG . 'cache/' . $key . '-persistent_data_cache-' . $this->getSecureToken()
		                 . '.pdc';
		
		return $cacheFilePath;
	}
	
	
	/**
	 * Returns the secure token.
	 *
	 * @return string
	 */
	protected function getSecureToken()
	{
		static $token;
		
		$dir = opendir(DIR_FS_CATALOG . 'media');
		if($dir !== false)
		{
			while(($file = readdir($dir)) !== false)
			{
				// search for secure token file
				if(strpos($file, 'secure_token_') !== false)
				{
					$token = str_replace('secure_token_', '', $file);
					break;
				}
			}
			
			if($token === null)
			{
				$token = md5(mt_rand());
			}
		}
		
		return $token;
	}
}
